class SnowJBSB extends TeamScoreBoard config nousercreate;


// ============================================================================
// Compiler Directives
// ============================================================================

#exec texture import name=JBScoreBoardElements file=Textures\ScoreBoardElements.pcx group=Icons flags=2 mips=off


// ============================================================================
// Enumerations
// ============================================================================

enum ESort {

  Sort_Score,
  Sort_Orders,
  Sort_Name,
  Sort_Ping,
  Sort_CountKillsAttack,
  Sort_CountKillsDefense,
  Sort_CountReleased,
  };


enum EStatus {

  Status_Entering,
  Status_Present,
  Status_Exiting,
  };


enum EIcon {

  Icon_None,
  Icon_Jailed,
  Icon_Llama,
  Icon_Arena,
  Icon_Protected,
  };


// ============================================================================
// Constants
// ============================================================================

const CountTeams = 2;


// ============================================================================
// Structures
// ============================================================================

struct TPosition {

  var int Column;
  var int Row;

  var Color ColorBig;
  var Color ColorSmall;
  var bool FlagOffscreen;
  };


struct TPoint {

  var int X;
  var int Y;
  };


struct TEntry {

  var int Id;
  var int Team;
  var int Score;
  var int ScoreGlobal;
  var int Ping;
  var int Time;
  var string Name;
  var string NamePrev;
  var string NameTruncated;
  var Texture Picture;
  var EIcon Icon;
  var bool FlagWaiting;

  var JBPRI JBInfo;
  var PlayerReplicationInfo Info;

  var int CountDeaths;
  var int CountKillsAttack;
  var int CountKillsDefense;
  var int CountReleased;
  var string TextLocation;
  var string TextLocationPrev;
  var string TextOrders;
  var string TextOrdersPrev;
  var string TextStatus;

  var TPosition Positions[3];
  var int CountPositions;
  var float RatioInterpolation;
  var float TimeInterpolation;
  
  var bool FlagPresent;
  var EStatus Status;
  
  var int RowPrev;
  };


// ============================================================================
// Configuration
// ============================================================================

var config ESort Sort;
var config bool NumericPlayerInfo;


// ============================================================================
// Properties
// ============================================================================

var() Color ColorTitle[2];
var() Color ColorGrid[2];
var() Color ColorStatsBig[2];
var() Color ColorStatsSmall[2];


// ============================================================================
// Variables
// ============================================================================

var JBPRI JBInfoPlayerLocal;
var JBSkinManager SkinManager;

var localized string TextBalanceNone;
var localized string TextBalanceLeader;
var localized string TextBalanceTrailer;
var localized string TextBalanceType[4];
var localized string TextBalanceBotSupported;
var localized string TextInfoSeparator;
var localized string TextInfoOvertime;
var localized string TextInfoCaptureCountdown;
var localized string TextInfoCaptureCountdownDisabled;
var localized string TextInfoCaptureTimeout;
var localized string TextLocationArena;
var localized string TextLocationJail[2];
var localized string TextLocationUndisclosed;
var localized string TextLocationUnknown;
var localized string TextStatsAttack;
var localized string TextStatsDefense;
var localized string TextStatsRelease;
var localized string TextSortLeader;
var localized string TextSortTrailer1;
var localized string TextSortTrailer2;
var localized string TextSortKey[7];

var float TimeRender;
var float TimeUpdate;

var float RatioFlash;
var Font FontTitle;
var Font FontSmall;
var Font FontSmallBold;
var Font FontStatsBig;
var Font FontStatsSmall;
var Font FontStatsSmallBold;
var Texture TextureSkinIconArena;
var Texture TextureSkinIconJailed;
var Texture TextureSkinIconLlama;
var Texture TextureSkinIconProtected;

var TEntry Entries[64];
var int CountEntries;
var int IndexEntries[32];

var TPoint PointOffsetColumns[2];
var TPoint PointOffsetMax;
var TPoint PointOffsetMin;

var int CountRows[2];
var int CountRowsPrev;
var int HeightEntry;
var int HeightEntryGap;
var int HeightScreen;
var int WidthEntry;
var int WidthEntryGap;
var int WidthEntryPrev;
var int WidthFrame;
var int WidthIcon;
var int WidthScreen;
var int WidthLine;
var float CountRowsCurrent;
var float CountRowsDelta;
var string TextKeySwitch;


// ============================================================================
// PreBeginPlay
// ============================================================================

event PreBeginPlay() {
  
  local int IndexColor;
  local int IndexIndex;
  
  SkinManager = class 'JBSkinManager'.static.GetManager(Self);

  SkinManager.GetSkin("ScoreBoardIcons");
  SkinManager.GetSkin("HudElements2");
  SkinManager.GetSkin("JailBreakLogo");
  
  for (IndexIndex = 0; IndexIndex < ArrayCount(IndexEntries); IndexIndex++)
    IndexEntries[IndexIndex] = -1;

  FontSmall     = Font(DynamicLoadObject("UWindowFonts.Tahoma10",  class 'Font', true));
  FontSmallBold = Font(DynamicLoadObject("UWindowFonts.TahomaB10", class 'Font', true));

  FontStatsSmall     = FontSmall;
  FontStatsSmallBold = FontSmallBold;
  
  for (IndexColor = 0; IndexColor < ArrayCount(ColorTitle); IndexColor++) {
    ColorTitle     [IndexColor] = TeamColor[IndexColor];
    ColorGrid      [IndexColor] = TeamColor[IndexColor];
    ColorStatsBig  [IndexColor] = TeamColor[IndexColor];
    ColorStatsSmall[IndexColor] = WhiteColor;
    }

  TextKeySwitch = FindKey("mutate utjb comp");

  Super.PreBeginPlay();
  }


// ============================================================================
// ShowScores
// ============================================================================

function ShowScores(Canvas Canvas) {

  local int CountRowsTarget;
  local int IndexColumn;
  local int IndexEntry;
  local int IndexTeam;
  local float HeightMove;
  local float HeightText;
  local float WidthText;
  local TPoint PointOffset;
  local string TextScore;
  local JBPRI ThisJBInfo;

  if (Owner.IsInState('Dying'))
    return;

  WidthScreen  = Canvas.ClipX;
  HeightScreen = Canvas.ClipY;

  PointOffsetMin.X = 0;
  PointOffsetMin.Y = 0;
  PointOffsetMax.X = WidthScreen;
  PointOffsetMax.Y = HeightScreen;

  TextureSkinIconArena     = SkinManager.GetSkin("Arena",     Texture 'Arena');
  TextureSkinIconJailed    = SkinManager.GetSkin("Jailed",    Texture 'Jailed');
  TextureSkinIconLlama     = SkinManager.GetSkin("Llama",     Texture 'Llama');
  TextureSkinIconProtected = SkinManager.GetSkin("Protected", Texture 'Protected');

  if (JBInfoPlayerLocal == None)
    foreach AllActors(class 'JBPRI', ThisJBInfo)
      if (ThisJBInfo.PRI == PlayerPawn(Owner).PlayerReplicationInfo)
        JBInfoPlayerLocal = ThisJBInfo;
  
  if (JBInfoPlayerLocal != None &&
      JBInfoPlayerLocal.FlagChangeCompass != JBInfoPlayerLocal.FlagChangeCompassPrev) {
    SortSwitch();
    JBInfoPlayerLocal.FlagChangeCompassPrev = JBInfoPlayerLocal.FlagChangeCompass;
    }

  WidthLine = 1;
  WidthFrame = 15;

  FontStatsBig = MyFonts.GetMediumFont(WidthScreen);

  Canvas.Font = FontStatsBig;
  Canvas.TextSize("X", WidthText, HeightText);

  RatioFlash -= ((Level.TimeSeconds - TimeRender) % 0.5) * 1.6;
  if (RatioFlash < 0.2)
    RatioFlash = 1.0;

  if (Level.TimeSeconds - TimeUpdate > 0.2) {
    EntriesMatch();
    EntriesSort();
    TimeUpdate = Level.TimeSeconds;
    }
  
  for (IndexColumn = 0; IndexColumn < ArrayCount(CountRows); IndexColumn++)
    CountRowsTarget = Max(CountRowsTarget, CountRows[IndexColumn]);

  if (CountRowsTarget == CountRowsPrev) {
    if (CountRowsDelta != 0.0) {
      CountRowsCurrent += CountRowsDelta * (Level.TimeSeconds - TimeRender) * 3.0;
  
      if (CountRowsDelta > 0.0)
        CountRowsCurrent = FMin(CountRowsTarget, CountRowsCurrent);
      else
        CountRowsCurrent = FMax(CountRowsTarget, CountRowsCurrent);
      
      if (CountRowsCurrent == CountRowsTarget)
        CountRowsDelta = 0.0;
      }
    }

  else {
    CountRowsDelta = CountRowsTarget - CountRowsCurrent;
    CountRowsPrev = CountRowsTarget;
    }

  //if (PlayerPawn(Owner).Weapon != None &&
  //    PlayerPawn(Owner).Weapon.bPointing &&
  //    PlayerPawn(Owner).MyHUD != None)
  //  PlayerPawn(Owner).MyHUD.DrawCrosshair(Canvas, WidthScreen / 2, HeightScreen / 2);

  DrawHeader(Canvas);
  DrawTrailer(Canvas);

  HeightEntry = HeightText;
  HeightEntryGap = Min(HeightText / 2, (PointOffsetMax.Y - PointOffsetMin.Y - 80) / CountRowsTarget - HeightEntry);
  
  WidthEntryPrev = WidthEntry;
  WidthEntry = Min(400, WidthScreen / CountTeams - WidthFrame * 2 - 10);
  WidthEntryGap = (WidthScreen - CountTeams * WidthEntry + WidthFrame * 2) / (CountTeams + 1);
  WidthIcon = HeightEntry;

  PointOffset = PointOffsetMin;
  PointOffset.X += WidthEntryGap - WidthFrame;
  PointOffset.Y += (PointOffsetMax.Y - PointOffsetMin.Y - CountRowsCurrent * (HeightEntry + HeightEntryGap) + HeightEntryGap) / 2;
  
  for (IndexColumn = 0; IndexColumn < ArrayCount(PointOffsetColumns); IndexColumn++) {
    PointOffsetColumns[IndexColumn] = PointOffset;
    PointOffset.X += WidthEntry + WidthEntryGap;
    }

  DrawTotals(Canvas);
  DrawGrid(Canvas);

  for (IndexEntry = 0; IndexEntry < CountEntries; IndexEntry++)
    EntryDraw(Canvas, Entries[IndexEntry]);

  Canvas.CurY = PointOffset.Y + CountRowsCurrent * (HeightEntry + HeightEntryGap) - HeightEntryGap + WidthFrame * 2;
  DrawSort(Canvas);

  TimeRender = Level.TimeSeconds;
  }


// ============================================================================
// Truncate
//
// Truncates the given text to the given pixel width, appending ellipsis if
// appropriate, and returns the result. Also optionally returns the width of
// the modified text.
// ============================================================================

function string Truncate(Canvas Canvas, string TextTruncate, float WidthTruncate, optional out float WidthText) {

  local float HeightText;
  local float LengthText;
  local int LengthTextMax;
  local int LengthTextMin;
  
  Canvas.StrLen(TextTruncate, WidthText, HeightText);
  if (WidthText <= WidthTruncate)
    return TextTruncate;
  
  LengthText = Len(TextTruncate);
  LengthTextMax = LengthText;
  
  while (LengthTextMax > LengthTextMin + 1) {
    Canvas.StrLen(Left(TextTruncate, LengthText) $ "...", WidthText, HeightText);

    if (WidthText > WidthTruncate)
      LengthTextMax = LengthText;
    else
      LengthTextMin = LengthText;
    
    LengthText = (LengthTextMin + LengthTextMax) / 2;
    }

  return Left(TextTruncate, LengthText) $ "...";
  }


// ============================================================================
// DrawBox
//
// Draws a box in the current color, starting and the current position, with
// the given width and height.
// ============================================================================

final function DrawBox(Canvas Canvas, int Width, int Height) {

  Canvas.bNoSmooth = true;

  if (Canvas.Style == ERenderStyle.STY_Modulated)
    Canvas.DrawTile(Texture 'JBScoreBoardElements', Width, Height, 1, 0, 1, 1);
  else
    Canvas.DrawTile(Texture 'JBScoreBoardElements', Width, Height, 0, 0, 1, 1);
  }


// ============================================================================
// DrawIcon
//
// Draws a Jailbreak icon at the current position with the current color and
// the given width and height.
// ============================================================================

function DrawIcon(Canvas Canvas, EIcon Icon, int Width, int Height) {

  local Texture TextureSkinIcon;

  switch (Icon) {
    case Icon_None:       return;
    case Icon_Arena:      TextureSkinIcon = TextureSkinIconArena;      break;
    case Icon_Jailed:     TextureSkinIcon = TextureSkinIconJailed;     break;
    case Icon_Llama:      TextureSkinIcon = TextureSkinIconLlama;      break;
    case Icon_Protected:  TextureSkinIcon = TextureSkinIconProtected;  break;
    }

  Canvas.bNoSmooth = false;
  Canvas.Style = ERenderStyle.STY_Translucent;

  Canvas.DrawTile(TextureSkinIcon, Width, Height, 0, TextureSkinIcon.VClamp - TextureSkinIcon.UClamp,
                                                  TextureSkinIcon.UClamp, TextureSkinIcon.UClamp);
  }


// ============================================================================
// DrawHeader
// ============================================================================

function DrawHeader(Canvas Canvas) {

  local int HeightLogo;
  local int WidthLogo;
  local GameReplicationInfo InfoGame;
  local Texture TextureLogo;

  InfoGame = TournamentGameReplicationInfo(PlayerPawn(Owner).GameReplicationInfo);

  Canvas.Reset();
  Canvas.bNoSmooth = false;
  Canvas.Style = ERenderStyle.STY_Translucent;
  Canvas.DrawColor.R = 255;
  Canvas.DrawColor.G = 255;
  Canvas.DrawColor.B = 255;

  TextureLogo = SkinManager.GetSkin("JailBreakLogo", Texture 'JailBreakLogo');

  WidthLogo = Min(TextureLogo.UClamp, WidthScreen * 256 / 800);
  HeightLogo = TextureLogo.VClamp * WidthLogo / TextureLogo.UClamp;

  Canvas.SetPos((WidthScreen - WidthLogo) / 2, 0);
  Canvas.DrawTile(TextureLogo, WidthLogo, HeightLogo, 0, 0, TextureLogo.UClamp, TextureLogo.VClamp);

  //if (Len(InfoGame.GameEndedComments) > 0) {
  //  Canvas.Style = ERenderStyle.STY_Normal;
  //  Canvas.Font = MyFonts.GetHugeFont(WidthScreen);
  //  Canvas.DrawColor = GoldColor;
  //  Canvas.SetPos(0, HeightLogo);
  //  Canvas.bCenter = true;
  //  Canvas.DrawText(InfoGame.GameEndedComments, true);
  //  }
  
  PointOffsetMin.Y = Max(PointOffsetMin.Y, HeightLogo);
  }


// ============================================================================
// DrawTrailer
// ============================================================================

function DrawTrailer(Canvas Canvas) {

  local int CountLines;
  local int TimeHours;
  local int TimeMinutes;
  local int TimeSeconds;
  local bool FlagOvertime;
  local float HeightText;
  local float WidthText;
  local string TextInfoGame;
  local string TextInfoLimits;
  local string TextInfoElapsed;
  local string TextInfoStatus;
  local TournamentGameReplicationInfo InfoGame;
  local JBGameReplicationInfo JBInfoGame;

  InfoGame = TournamentGameReplicationInfo(PlayerPawn(Owner).GameReplicationInfo);
  JBInfoGame = JBGameReplicationInfo(InfoGame);

  FlagOvertime = InfoGame.TimeLimit > 0 && JBInfoGame.JBElapsed > InfoGame.TimeLimit * 60;

  if (Level.NetMode == NM_Standalone) {
    if (DeathMatchPlus(Level.Game).bRatedGame)
      TextInfoGame = DeathMatchPlus(Level.Game).RatedGameLadderObj.SkillText;
    else if (DeathMatchPlus(Level.Game).bNoviceMode)
      TextInfoGame = class 'ChallengeBotInfo'.default.Skills[Level.Game.Difficulty];
    else
      TextInfoGame = class 'ChallengeBotInfo'.default.Skills[Level.Game.Difficulty + 4];

    TextInfoGame = TextInfoGame @ InfoGame.GameName @ MapTitle @ MapTitleQuote $ Level.Title $ MapTitleQuote;
	}

  else
    TextInfoGame = InfoGame.GameName @ MapTitle @ Level.Title;

  if (InfoGame.GoalTeamScore > 0)
    TextInfoLimits = FragGoal @ InfoGame.GoalTeamScore;

  if (InfoGame.TimeLimit > 0) {
    if (Len(TextInfoLimits) > 0)
      TextInfoLimits = TextInfoLimits $ TextInfoSeparator;
    TextInfoLimits = TextInfoLimits $ TimeLimit @ InfoGame.TimeLimit $ ":00";
    }

  if (JBGameReplicationInfo(InfoGame).MaxCapTime > 0) {
    if (Len(TextInfoLimits) > 0)
      TextInfoLimits = TextInfoLimits $ TextInfoSeparator;
    TextInfoLimits = TextInfoLimits $ TextInfoCaptureTimeout @ int(JBInfoGame.MaxCapTime) $ ":00";
    }

  if (bTimeDown || InfoGame.RemainingTime > 0) {
    bTimeDown = true;

    TimeSeconds = Abs(InfoGame.TimeLimit * 60 - JBInfoGame.JBElapsed);

    TimeMinutes = TimeSeconds / 60;
    TimeSeconds = TimeSeconds % 60;

    TextInfoElapsed = RemainingTime @ TimeMinutes $ ":" $ TwoDigitString(TimeSeconds);
    if (FlagOvertime)
      TextInfoElapsed = TextInfoElapsed @ TextInfoOvertime;
	}

  else {
    TimeSeconds = JBInfoGame.JBElapsed;
    TimeMinutes = TimeSeconds / 60;
    TimeHours   = TimeMinutes / 60;
    TimeSeconds = TimeSeconds - TimeMinutes * 60;
    TimeMinutes = TimeMinutes - TimeHours   * 60;

    TextInfoElapsed = ElapsedTime @ TwoDigitString(TimeHours)   $ ":" $
                                    TwoDigitString(TimeMinutes) $ ":" $
                                    TwoDigitString(TimeSeconds);
	}

  if (JBInfoGame.MaxCapTime > 0 && Len(InfoGame.GameEndedComments) == 0) {
    TimeSeconds = JBInfoGame.NextCapTime - JBInfoGame.JBElapsed;
    TimeMinutes = TimeSeconds / 60;
    TimeSeconds = TimeSeconds % 60;

    TextInfoElapsed = TextInfoElapsed $ TextInfoSeparator $ TextInfoCaptureCountdown;

    if (FlagOvertime)
      TextInfoElapsed = TextInfoElapsed @ TextInfoCaptureCountdownDisabled;
    else
      TextInfoElapsed = TextInfoElapsed @ TimeMinutes $ ":" $ TwoDigitString(TimeSeconds);
    }

  CountLines = 2;
  if (Len(TextInfoLimits) > 0)
    CountLines++;

  Canvas.Reset();
  Canvas.bNoSmooth = false;
  Canvas.bCenter = true;

  Canvas.DrawColor = WhiteColor;
  Canvas.Font = MyFonts.GetSmallFont(WidthScreen);
  Canvas.StrLen("X", WidthText, HeightText);
  Canvas.SetPos(0, HeightScreen - HeightText * CountLines);

  PointOffsetMax.Y = Min(PointOffsetMax.Y, Canvas.CurY);

  if (PlayerPawn(Owner).ViewTarget != None &&
      PlayerPawn(Owner).ViewTarget.IsA('JBArenaView'))
    return;

  Canvas.DrawText(TextInfoGame,    true);  Canvas.CurY -= HeightText;
  Canvas.DrawText(TextInfoLimits,  true);  Canvas.CurY -= HeightText;
  Canvas.DrawText(TextInfoElapsed, true);

  if (Len(InfoGame.GameEndedComments) > 0) {
    if (Level.NetMode == NM_Standalone)
      TextInfoStatus = Ended @ Continue;
    else
      TextInfoStatus = Ended;
    }

  else if (PlayerPawn(Owner) != None && PlayerPawn(Owner).Health <= 0) 
    TextInfoStatus = Restart;

  if (Len(TextInfoStatus) > 0) {
    Canvas.DrawColor = GreenColor;
    Canvas.SetPos(0, HeightScreen - HeightText * (CountLines + 2));
    Canvas.DrawText(TextInfoStatus, true);
    }
  }


// ============================================================================
// DrawTotals
//
// Draws the total team scores.
// ============================================================================

function DrawTotals(Canvas Canvas) {

  local int IndexColumn;
  local float HeightText;
  local float WidthText;
  local string TextScore;

  Canvas.Reset();
  Canvas.bNoSmooth = false;
  
  Canvas.Font = MyFonts.GetHugeFont(WidthScreen);

  for (IndexColumn = 0; IndexColumn < ArrayCount(PointOffsetColumns); IndexColumn++) {
    TextScore = string(int(TournamentGameReplicationInfo(PlayerPawn(Owner).GameReplicationInfo).Teams[IndexColumn].Score));
  
    Canvas.TextSize(TextScore, WidthText, HeightText);
    Canvas.DrawColor = TeamColor[IndexColumn];

    Canvas.SetPos(PointOffsetColumns[IndexColumn].X,
                  PointOffsetColumns[IndexColumn].Y - WidthFrame - HeightText);

    Canvas.DrawText(TeamName[IndexColumn], false);

    Canvas.SetPos(PointOffsetColumns[IndexColumn].X + WidthEntry - WidthText,
                  PointOffsetColumns[IndexColumn].Y - WidthFrame - HeightText);

    Canvas.DrawText(TextScore, false);
    }
  }


// ============================================================================
// DrawGrid
//
// Draws the grid.
// ============================================================================

function DrawGrid(Canvas Canvas) {

  local int CountRowsMax;
  local int IndexRow;
  local int IndexColumn;
  local TPoint PointOffset;
  
  for (IndexColumn = 0; IndexColumn < ArrayCount(CountRows); IndexColumn++) {
    PointOffset = PointOffsetColumns[IndexColumn];
    PointOffset.Y += HeightEntry; 
    
    Canvas.Style = ERenderStyle.STY_Modulated;
    Canvas.DrawColor.R = 0;
    Canvas.DrawColor.G = 0;
    Canvas.DrawColor.B = 0;
    
    Canvas.SetPos(PointOffsetColumns[IndexColumn].X - 15, PointOffsetColumns[IndexColumn].Y - 15);
    DrawBox(Canvas, WidthEntry + 30, CountRowsCurrent * (HeightEntry + HeightEntryGap) - HeightEntryGap + 30);
    
    Canvas.Style = ERenderStyle.STY_Normal;
    Canvas.DrawColor = ColorGrid[IndexColumn];

    for (IndexRow = 0; IndexRow < CountRows[IndexColumn]; IndexRow++) {
      if (IndexRow > CountRowsCurrent)
        break;
    
      Canvas.SetPos(PointOffset.X, PointOffset.Y);
      DrawBox(Canvas, WidthEntry, WidthLine);

      PointOffset.Y += HeightEntry + HeightEntryGap;
      }
    }
  }


// ============================================================================
// DrawSort
//
// Draws the sort key information at the current vertical position.
// ============================================================================

function DrawSort(Canvas Canvas) {

  local float HeightText;
  local float WidthText;
  local int IndexBalance;
  local int WidthTextTotal;
  local string TextBalanceNormal;
  local string TextBalanceBold;
  local string TextSortTrailer;
  local JBGameReplicationInfo JBInfoGame;

  JBInfoGame = JBGameReplicationInfo(PlayerPawn(Owner).GameReplicationInfo);
  IndexBalance = JBInfoGame.TeamBalance;

  if (IndexBalance == 0) {
    TextBalanceNormal = TextBalanceNone;
    TextBalanceBold = "";
    }

  else {
    TextBalanceNormal = TextBalanceLeader $ TextBalanceTrailer;
    TextBalanceBold = TextBalanceType[IndexBalance];
    if (JBInfoGame.FlagTeamBalanceBotSupported)
      TextBalanceBold = TextBalanceBold $ TextBalanceBotSupported;
    }

  TextSortTrailer = TextSortTrailer1 $ TextKeySwitch $ TextSortTrailer2;

  Canvas.Font = FontSmall;
  Canvas.TextSize(TextSortLeader $ TextSortTrailer $ TextBalanceNormal, WidthText, HeightText);
  WidthTextTotal = WidthText;

  Canvas.Font = FontSmallBold;
  Canvas.TextSize(TextSortKey[Sort] $ TextBalanceBold, WidthText, HeightText);
  WidthTextTotal += WidthText;

  Canvas.bNoSmooth = false;
  Canvas.Style = ERenderStyle.STY_Normal;
  Canvas.DrawColor = WhiteColor;

  Canvas.CurX = (WidthScreen - WidthTextTotal) / 2;

  Canvas.Font = FontSmall;
  Canvas.DrawText(TextSortLeader, false);

  Canvas.Font = FontSmallBold;
  Canvas.CurY -= HeightText;
  Canvas.DrawText(TextSortKey[Sort], false);

  Canvas.Font = FontSmall;
  Canvas.CurY -= HeightText;
  Canvas.DrawText(TextSortTrailer, false);

  if (IndexBalance == 0) {
    Canvas.CurY -= HeightText;
    Canvas.DrawText(TextBalanceNone, false);
    }
    
  else {
    Canvas.CurY -= HeightText;
    Canvas.DrawText(TextBalanceLeader, false);
    
    Canvas.Font = FontSmallBold;
    Canvas.CurY -= HeightText;
    Canvas.DrawText(TextBalanceType[IndexBalance], false);
    
    if (JBInfoGame.FlagTeamBalanceBotSupported) {
      Canvas.CurY -= HeightText;
      Canvas.DrawText(TextBalanceBotSupported, false);
      }
    
    Canvas.Font = FontSmall;
    Canvas.CurY -= HeightText;
    Canvas.DrawText(TextBalanceTrailer, false);
    }
  }


// ============================================================================
// SortSwitch
//
// Changes score sorting.
// ============================================================================

function SortSwitch() {

  switch (Sort) {
    case Sort_Score:              Sort = Sort_Orders;             break;
    case Sort_Orders:             Sort = Sort_Name;               break;
    case Sort_Name:               Sort = Sort_Ping;               break;
    case Sort_Ping:               Sort = Sort_CountKillsAttack;   break;
    case Sort_CountKillsAttack:   Sort = Sort_CountKillsDefense;  break;
    case Sort_CountKillsDefense:  Sort = Sort_CountReleased;      break;
    case Sort_CountReleased:      Sort = Sort_Score;              break;
    }
  }


// ============================================================================
// FindKey
//
// Determines the key name that's bound to the given command. Partial commands
// match.
// ============================================================================

function string FindKey(string TextCommand) {

  local int IndexKey;
  local string TextKey;
  local string TextAlias;
  
  for (IndexKey = 0; IndexKey < 256; IndexKey++) {
    TextKey = PlayerPawn(Owner).ConsoleCommand("KEYNAME" @ IndexKey);
    if (Len(TextKey) == 0)
      continue;
    
    TextAlias = PlayerPawn(Owner).ConsoleCommand("KEYBINDING" @ TextKey);
    
    if (InStr(Caps(TextAlias), Caps(TextCommand)) >= 0)
      return TextKey;
    }

  return "";
  }


// ============================================================================
// ConvertPositionPoint
//
// Converts a logical position to an actual pixel value.
// ============================================================================

function TPoint ConvertPositionPoint(TPosition Position) {

  local TPoint PointResult;
  
  PointResult.Y = PointOffsetColumns[Position.Column].Y + Position.Row * (HeightEntry + HeightEntryGap);
  
  if (!Position.FlagOffscreen)
    PointResult.X = PointOffsetColumns[Position.Column].X;
  else
    if (Position.Column * 2 < ArrayCount(PointOffsetColumns))
      PointResult.X = -WidthEntry;
    else
      PointResult.X = WidthScreen;
    
  return PointResult;
  }


// ============================================================================
// EntryDraw
//
// Draws an entry to the given canvas.
// ============================================================================

function EntryDraw(Canvas Canvas, out TEntry Entry) {

  local bool FlagEnded;
  local int ColorDeltaR;
  local int ColorDeltaG;
  local int ColorDeltaB;
  local int CountPing;
  local int IndexPosition;
  local int SizePing;
  local int SizePingMax;
  local TPoint PointInterpolated;
  local TPoint PointInterpolationStart;
  local TPoint PointInterpolationEnd;
  local TPoint VectorDelta;
  local float HeightAmplitude;
  local float HeightText;
  local float WidthText;
  local float WidthTextScore;
  local float WidthTextStatus;
  local Color ColorBigInterpolated;
  local Color ColorSmallInterpolated;
  local string TextStatus;
  local string TextTime;
  
  FlagEnded = Len(PlayerPawn(Owner).GameReplicationInfo.GameEndedComments) > 0;
  
  Canvas.bNoSmooth = false;
  Canvas.Style = ERenderStyle.STY_Normal;

  if (Entry.CountPositions > 1) {
    Entry.RatioInterpolation += (Level.TimeSeconds - Entry.TimeInterpolation) * 3.0;
    if (Entry.RatioInterpolation > 1.0)
      Entry.RatioInterpolation = 1.0;
  
    PointInterpolationStart = ConvertPositionPoint(Entry.Positions[0]);
    PointInterpolationEnd   = ConvertPositionPoint(Entry.Positions[1]);

    VectorDelta.X = PointInterpolationEnd.X - PointInterpolationStart.X;
    VectorDelta.Y = PointInterpolationEnd.Y - PointInterpolationStart.Y;

    PointInterpolated = PointInterpolationStart;
    PointInterpolated.X += Entry.RatioInterpolation * VectorDelta.X; 
    PointInterpolated.Y += Entry.RatioInterpolation * VectorDelta.Y;

    if (Entry.Positions[0].Column == Entry.Positions[1].Column &&
        !Entry.Positions[0].FlagOffscreen &&
        !Entry.Positions[1].FlagOffscreen) {

      HeightAmplitude = HeightEntry * (1.0 - Square(Entry.RatioInterpolation * 2.0 - 1.0)) /
                        Sqrt(Square(VectorDelta.X) + Square(VectorDelta.Y));
      PointInterpolated.X += VectorDelta.Y * HeightAmplitude;
      PointInterpolated.Y -= VectorDelta.X * HeightAmplitude;
      }

    ColorDeltaR = Entry.Positions[1].ColorBig.R - Entry.Positions[0].ColorBig.R;
    ColorDeltaG = Entry.Positions[1].ColorBig.G - Entry.Positions[0].ColorBig.G;
    ColorDeltaB = Entry.Positions[1].ColorBig.B - Entry.Positions[0].ColorBig.B;
    ColorBigInterpolated.R = Entry.Positions[0].ColorBig.R + ColorDeltaR * Entry.RatioInterpolation;
    ColorBigInterpolated.G = Entry.Positions[0].ColorBig.G + ColorDeltaG * Entry.RatioInterpolation;
    ColorBigInterpolated.B = Entry.Positions[0].ColorBig.B + ColorDeltaB * Entry.RatioInterpolation;

    ColorDeltaR = Entry.Positions[1].ColorSmall.R - Entry.Positions[0].ColorSmall.R;
    ColorDeltaG = Entry.Positions[1].ColorSmall.G - Entry.Positions[0].ColorSmall.G;
    ColorDeltaB = Entry.Positions[1].ColorSmall.B - Entry.Positions[0].ColorSmall.B;
    ColorSmallInterpolated.R = Entry.Positions[0].ColorSmall.R + ColorDeltaR * Entry.RatioInterpolation;
    ColorSmallInterpolated.G = Entry.Positions[0].ColorSmall.G + ColorDeltaG * Entry.RatioInterpolation;
    ColorSmallInterpolated.B = Entry.Positions[0].ColorSmall.B + ColorDeltaB * Entry.RatioInterpolation;

    if (Entry.RatioInterpolation == 1.0) {
      for (IndexPosition = 1; IndexPosition < Entry.CountPositions; IndexPosition++)
        Entry.Positions[IndexPosition - 1] = Entry.Positions[IndexPosition];
      Entry.CountPositions--;
      Entry.RatioInterpolation = 0.0;
      }
    }
  
  else {
    PointInterpolated      = ConvertPositionPoint(Entry.Positions[0]);
    ColorBigInterpolated   = Entry.Positions[0].ColorBig;
    ColorSmallInterpolated = Entry.Positions[0].ColorSmall;
    }

  Entry.TimeInterpolation = Level.TimeSeconds;

  Canvas.SetPos(PointInterpolated.X, PointInterpolated.Y - HeightEntry / 10);

  if (Entry.Icon == Icon_None && Entry.Picture != None) {
    Canvas.bNoSmooth = false;
    Canvas.Style = ERenderStyle.STY_Translucent;
    Canvas.DrawColor.R = 255;
    Canvas.DrawColor.G = 255;
    Canvas.DrawColor.B = 255;
    Canvas.DrawTile(Entry.Picture, WidthIcon, WidthIcon, 0, 0, Entry.Picture.UClamp, Entry.Picture.VClamp);
    }
  
  else {
    Canvas.DrawColor = ColorBigInterpolated;
    DrawIcon(Canvas, Entry.Icon, WidthIcon, WidthIcon);
    }

  Canvas.Font = FontStatsBig;
  Canvas.DrawColor = ColorBigInterpolated;

  if (WidthEntry != WidthEntryPrev || Entry.Name != Entry.NamePrev) {
    Entry.NameTruncated = Truncate(Canvas, Entry.Name, WidthEntry * 0.45 - WidthIcon * 1.2);
    Entry.NamePrev = Entry.Name;
    }

  if (Entry.FlagWaiting) {  
    Canvas.DrawColor.R = Canvas.DrawColor.R * RatioFlash;
    Canvas.DrawColor.G = Canvas.DrawColor.G * RatioFlash;
    Canvas.DrawColor.B = Canvas.DrawColor.B * RatioFlash;
    }

  Canvas.SetPos(PointInterpolated.X + WidthIcon * 1.2, PointInterpolated.Y);
  Canvas.DrawTextClipped(Entry.NameTruncated, false);

  if (Entry.ScoreGlobal > 0) {
    Canvas.Font = FontStatsSmallBold;
    Canvas.TextSize(Entry.ScoreGlobal, WidthText, HeightText);
    Canvas.SetPos(PointInterpolated.X + WidthEntry * 0.48 - WidthLine * 3 - WidthText,
                  PointInterpolated.Y + HeightEntry - HeightText * 1.2);
    Canvas.DrawTextClipped(Entry.ScoreGlobal, false);
    }

  Canvas.Font = FontStatsBig;
  Canvas.TextSize(Entry.Score, WidthTextScore, HeightText);
  Canvas.SetPos(PointInterpolated.X + WidthEntry - WidthTextScore, PointInterpolated.Y);
  Canvas.DrawTextClipped(Entry.Score, false);

  Canvas.TextSize("00", WidthText, HeightText);
  if (WidthText > WidthTextScore)
    WidthTextScore = WidthText;

  Canvas.Font = FontStatsSmall;
  Canvas.DrawColor = ColorSmallInterpolated;
  Canvas.TextSize("X", WidthText, HeightText);

  if (FlagEnded || JBViewPoint(PlayerPawn(Owner).ViewTarget) != None) {
    Entry.TextStatus = "";
    
    if (Entry.CountKillsAttack > 0)
      Entry.TextStatus = Entry.CountKillsAttack @ TextStatsAttack;

    if (Entry.CountKillsDefense > 0) {
      if (Len(Entry.TextStatus) > 0)
        Entry.TextStatus = Entry.TextStatus $ ", ";
      Entry.TextStatus = Entry.TextStatus $ Entry.CountKillsDefense @ TextStatsDefense;
      }

    if (Entry.CountReleased > 0) {
      if (Len(Entry.TextStatus) > 0)
        Entry.TextStatus = Entry.TextStatus $ ", ";
      Entry.TextStatus = Entry.TextStatus $ Entry.CountReleased @ TextStatsRelease;
      }

    Entry.TextLocationPrev = "";
    Entry.TextOrdersPrev   = "";
    }

  else if (WidthEntry         != WidthEntryPrev ||
           Entry.TextLocation != Entry.TextLocationPrev ||
           Entry.TextOrders   != Entry.TextOrdersPrev) {

    Entry.TextStatus = Entry.TextLocation;

    if (Len(Entry.TextOrders) > 0 && Mid(Entry.TextOrders, 0, 1) != "*") {
      if (Len(Entry.TextStatus) > 0)
        Entry.TextStatus = Entry.TextStatus $ " ";
      Entry.TextStatus = Entry.TextStatus $ "(" $ Entry.TextOrders $ ")";
      }
    
    Entry.TextStatus = Truncate(Canvas, Entry.TextStatus, WidthEntry * 0.5 - WidthTextScore - 10);

    Entry.TextLocationPrev = Entry.TextLocation;
    Entry.TextOrdersPrev   = Entry.TextOrders;
    }
  
  if (Len(Entry.TextStatus) > 0) {
    Canvas.SetPos(PointInterpolated.X + WidthEntry * 0.5,
                  PointInterpolated.Y + HeightEntry - HeightText * 1.2);
    Canvas.DrawTextClipped(Entry.TextStatus);
    }

  if (Entry.Ping > 0) {
    CountPing = (Clamp(Entry.Ping, 50, 300) - 50) * 4;
  
    Canvas.DrawColor.R = 192 *         Min(500, CountPing)  / 500;
    Canvas.DrawColor.G = 192 * (1000 - Max(500, CountPing)) / 500;
    Canvas.DrawColor.B = 0;

    SizePingMax = HeightEntry - 6;
    SizePing = Min(SizePingMax, Entry.Ping * SizePingMax / 500);

    Canvas.Style = ERenderStyle.STY_Normal;
    Canvas.SetPos(PointInterpolated.X + WidthEntry * 0.49 - WidthLine * 3,
                  PointInterpolated.Y + SizePingMax - SizePing);
    DrawBox(Canvas, WidthLine * 3, SizePing);
    }

  if (NumericPlayerInfo && Level.NetMode != NM_Standalone && !Entry.JBInfo.PRI.bIsABot) {
    if (Entry.Ping > 0) {
      Canvas.Font = FontStatsSmall;
      Canvas.DrawColor = WhiteColor * 0.3;
      Canvas.Style = ERenderStyle.STY_Translucent;

      Canvas.TextSize(Entry.Ping, WidthText, HeightText);
      Canvas.SetPos(PointInterpolated.X + WidthEntry * 0.98 - WidthTextScore - WidthText,
                    PointInterpolated.Y + HeightEntry - HeightText * 2.2);
      Canvas.DrawTextClipped(Entry.Ping);
      }

    TextTime = string(int(Entry.Time % 60));
    if (Len(TextTime) < 2)
      TextTime = "0" $ TextTime;
    TextTime = (Entry.Time / 60) $ ":" $ TextTime;
    
    Canvas.Font = FontStatsSmall;
    Canvas.DrawColor = WhiteColor * 0.3;
    Canvas.Style = ERenderStyle.STY_Translucent;

    Canvas.TextSize(TextTime, WidthText, HeightText);
    Canvas.SetPos(PointInterpolated.X + WidthEntry * 0.98 - WidthTextScore - WidthText,
                  PointInterpolated.Y + HeightEntry - HeightText * 1.2);
    Canvas.DrawTextClipped(TextTime);
    }
  }


// ============================================================================
// EntryBelow
//
// Compares two entries according to the current sort key and returns true if
// the first entry is to be sorted below the second entry.
// ============================================================================

final function bool EntryBelow(TEntry Entry1, TEntry Entry2) {

  switch (Sort) {
    case Sort_Score:
      return (Entry1.Score <  Entry2.Score ||
             (Entry1.Score == Entry2.Score &&
               (Entry1.RowPrev > Entry2.RowPrev)));
    
    case Sort_Orders:
      if (Entry1.Icon == Icon_Arena || Entry1.Icon == Icon_Jailed)
        if (Entry2.Icon == Icon_Arena || Entry2.Icon == Icon_Jailed)
          return (Entry1.Name > Entry2.Name);
        else
          return true;
      if (Entry2.Icon == Icon_Arena || Entry2.Icon == Icon_Jailed)
        return false;

      if (!JBInfoPlayerLocal.FlagSupervisor &&
          Pawn(Owner).PlayerReplicationInfo.Team != 255 &&
          Entry1.Team != Pawn(Owner).PlayerReplicationInfo.Team &&
          Entry2.Team != Pawn(Owner).PlayerReplicationInfo.Team)
        return (Entry1.Name > Entry2.Name);

      return (Entry1.TextOrders >  Entry2.TextOrders ||
             (Entry1.TextOrders == Entry2.TextOrders &&
               (Entry1.Name > Entry2.Name)));
    
    case Sort_Name:
      return (Entry1.Name > Entry2.Name);
    
    case Sort_Ping:
      return (Entry1.Ping >  Entry2.Ping ||
             (Entry1.Ping == Entry2.Ping &&
               (Entry1.RowPrev > Entry2.RowPrev)));
    
    case Sort_CountKillsAttack:
      return (Entry1.CountKillsAttack <  Entry2.CountKillsAttack ||
             (Entry1.CountKillsAttack == Entry2.CountKillsAttack &&
               (Entry1.RowPrev > Entry2.RowPrev)));

    case Sort_CountKillsDefense:
      return (Entry1.CountKillsDefense <  Entry2.CountKillsDefense ||
             (Entry1.CountKillsDefense == Entry2.CountKillsDefense &&
               (Entry1.RowPrev > Entry2.RowPrev)));

    case Sort_CountReleased:
      return (Entry1.CountReleased <  Entry2.CountReleased ||
             (Entry1.CountReleased == Entry2.CountReleased &&
               (Entry1.RowPrev > Entry2.RowPrev)));
    }
  }


// ============================================================================
// EntrySwap
//
// Swaps two entries.
// ============================================================================

final function EntrySwap(out TEntry Entry1, out TEntry Entry2) {

  local TEntry EntryTemp;
  
  EntryTemp = Entry1;
  Entry1 = Entry2;
  Entry2 = EntryTemp;
  }


// ============================================================================
// EntryMove
//
// Initiates an animated move of the given entry to the given new position.
// ============================================================================

function EntryMove(out TEntry Entry, TPosition Position) {

  if (Entry.CountPositions > 0 && Entry.Positions[Entry.CountPositions - 1] == Position)
    return;

  if (Entry.CountPositions == ArrayCount(Entry.Positions))
    Entry.Positions[Entry.CountPositions - 1] = Position;
  else
    Entry.Positions[Entry.CountPositions++] = Position;
  }


// ============================================================================
// EntriesMatch
//
// Iterates over all players in the current game, matching existing entries
// with players, initiating exit animations for players who left and adding
// new entries for new players. Updates entry content.
// ============================================================================

function EntriesMatch() {

  local bool FlagArena;
  local bool FlagJailed;
  local bool FlagDisclose;
  local int IndexEntry;
  local int IndexInfoPlayer;
  local GameReplicationInfo InfoGame;
  local PlayerReplicationInfo InfoPlayer;
  local JBPRI JBInfoPlayer;
  local JBPRI ThisJBInfo;
  local TPosition PositionEntry;
  
  for (IndexEntry = 0; IndexEntry < CountEntries; IndexEntry++)
    Entries[IndexEntry].FlagPresent = false;

  InfoGame = PlayerPawn(Owner).GameReplicationInfo;
  if (InfoGame == None)
    return;

  for (IndexInfoPlayer = 0; IndexInfoPlayer < ArrayCount(InfoGame.PRIArray); IndexInfoPlayer++) {
    InfoPlayer = InfoGame.PRIArray[IndexInfoPlayer];
    if (InfoPlayer == None || InfoPlayer.Team >= CountTeams || (InfoPlayer.bIsSpectator && !InfoPlayer.bWaitingPlayer))
      continue;
    
    IndexEntry = IndexEntries[InfoPlayer.PlayerID];

    if (IndexEntry < 0 || Entries[IndexEntry].Info != InfoPlayer) {
      IndexEntry = CountEntries;
      CountEntries++;
    
      Entries[IndexEntry].Id      = InfoPlayer.PlayerID;
      Entries[IndexEntry].Info    = InfoPlayer;
      Entries[IndexEntry].JBInfo  = None;
      Entries[IndexEntry].Status  = Status_Entering;
      Entries[IndexEntry].RowPrev = IndexEntry;
      Entries[IndexEntry].CountPositions = 0;

      IndexEntries[InfoPlayer.PlayerID] = IndexEntry;
      }

    if (Entries[IndexEntry].JBInfo == None)
      foreach AllActors(class 'JBPRI', ThisJBInfo)
        if (InfoPlayer == ThisJBInfo.PRI)
          Entries[IndexEntry].JBInfo = ThisJBInfo;

    JBInfoPlayer = Entries[IndexEntry].JBInfo;

    if (InfoPlayer.PlayerName != "") {
      Entries[IndexEntry].Team    = InfoPlayer.Team;
      Entries[IndexEntry].Score   = InfoPlayer.Score;
      Entries[IndexEntry].Ping    = InfoPlayer.Ping;
      Entries[IndexEntry].Name    = InfoPlayer.PlayerName;
      Entries[IndexEntry].Picture = InfoPlayer.TalkTexture;
      Entries[IndexEntry].Time    = Level.TimeSeconds + Pawn(Owner).PlayerReplicationInfo.StartTime - InfoPlayer.StartTime;

      if (JBInfoPlayer != None) {
        if (JBInfoPlayer.IsInArena)
          Entries[IndexEntry].Icon = Icon_Arena;
        else if (JBInfoPlayer.IsProtected)
          Entries[IndexEntry].Icon = Icon_Protected;
        else if (JBInfoPlayer.IsJailed)
          Entries[IndexEntry].Icon = Icon_Jailed;
        else if (JBInfoPlayer.LlamaHunt > InfoGame.ElapsedTime)
          Entries[IndexEntry].Icon = Icon_Llama;
        else
          Entries[IndexEntry].Icon = Icon_None;

        Entries[IndexEntry].FlagWaiting       = JBInfoPlayer.IsWaiting;
        Entries[IndexEntry].ScoreGlobal       = JBInfoPlayer.GlobalStanding;
        Entries[IndexEntry].CountKillsAttack  = JBInfoPlayer.CountKillsAttack;
        Entries[IndexEntry].CountKillsDefense = JBInfoPlayer.CountKillsDefense;
        Entries[IndexEntry].CountReleased     = JBInfoPlayer.CountReleased;
        }

      Entries[IndexEntry].CountDeaths = InfoPlayer.Deaths;
      
      FlagArena    = JBInfoPlayer != None && JBInfoPlayer.IsInArena;
      FlagJailed   = JBInfoPlayer != None && JBInfoPlayer.IsJailed;
      FlagDisclose = FlagArena || FlagJailed || JBInfoPlayerLocal.FlagSupervisor ||
                     PlayerPawn(Owner).PlayerReplicationInfo.Team == InfoPlayer.Team ||
                     Spectator(Owner) != None;
      
      if (FlagDisclose && InfoPlayer.PlayerLocation != None && Len(InfoPlayer.PlayerLocation.LocationName) > 0)
        Entries[IndexEntry].TextLocation = InfoPlayer.PlayerLocation.LocationName;
      else if (FlagDisclose && InfoPlayer.PlayerZone != None && Len(InfoPlayer.PlayerZone.ZoneName) > 0)
        Entries[IndexEntry].TextLocation = InfoPlayer.PlayerZone.ZoneName;
      else if (FlagArena)
        Entries[IndexEntry].TextLocation = TextLocationArena;
      else if (FlagJailed)
        Entries[IndexEntry].TextLocation = TextLocationJail[1 - InfoPlayer.Team];
      else if (FlagDisclose)
        Entries[IndexEntry].TextLocation = TextLocationUnknown;
      else
        Entries[IndexEntry].TextLocation = TextLocationUndisclosed;
      
      if (FlagDisclose && !FlagJailed && !FlagArena)
        Entries[IndexEntry].TextOrders = TournamentGameReplicationInfo(InfoGame).GetOrderString(InfoPlayer);
      else
        Entries[IndexEntry].TextOrders = "";

      while (Right(Entries[IndexEntry].TextOrders, 1) == " ")
        Entries[IndexEntry].TextOrders = Left(Entries[IndexEntry].TextOrders, Len(Entries[IndexEntry].TextOrders) - 1);
      }
    
    Entries[IndexEntry].FlagPresent = true;
    }

  for (IndexEntry = 0; IndexEntry < CountEntries; IndexEntry++) {
    if (Entries[IndexEntry].FlagPresent)
      continue;
      
    if (Entries[IndexEntry].Status == Status_Exiting) {
      if (Entries[IndexEntry].CountPositions == 1) {
        Entries[IndexEntry] = Entries[--CountEntries];
        IndexEntry--;
        }
      }
    
    else {
      PositionEntry = Entries[IndexEntry].Positions[Entries[IndexEntry].CountPositions - 1];
      PositionEntry.FlagOffscreen = true;

      EntryMove(Entries[IndexEntry], PositionEntry);
      Entries[IndexEntry].Status = Status_Exiting;

      IndexEntries[Entries[IndexEntry].Id] = -1;
      }
    }
  }


// ============================================================================
// EntriesSort
//
// Sorts all entries and initiates animations to move them.
// ============================================================================

function EntriesSort() {

  local int IndexColumn;
  local int IndexEntry;
  local int IndexEntry1;
  local int IndexEntry2;
  local TPosition PositionEntry;
  local TPosition PositionEntryEntering;

  for (IndexColumn = 0; IndexColumn < ArrayCount(CountRows); IndexColumn++)
    CountRows[IndexColumn] = 0;

  for (IndexEntry1 = 0; IndexEntry1 < CountEntries - 1; IndexEntry1++)
    for (IndexEntry2 = CountEntries - 1; IndexEntry2 > IndexEntry1; IndexEntry2--)
      if (EntryBelow(Entries[IndexEntry1], Entries[IndexEntry2]))
        EntrySwap(Entries[IndexEntry1], Entries[IndexEntry2]);

  for (IndexEntry = 0; IndexEntry < CountEntries; IndexEntry++) {
    if (Entries[IndexEntry].Status == Status_Exiting)
      continue;
  
    IndexEntries[Entries[IndexEntry].Id] = IndexEntry;

    PositionEntry.Column = Entries[IndexEntry].Team;
    PositionEntry.Row = CountRows[PositionEntry.Column];
    PositionEntry.ColorSmall = ColorStatsSmall[PositionEntry.Column];

    Entries[IndexEntry].RowPrev = CountRows[PositionEntry.Column];

    if (Entries[IndexEntry].Info.bAdmin)
      PositionEntry.ColorBig = WhiteColor;
    else if (Entries[IndexEntry].Name == PlayerPawn(Owner).PlayerReplicationInfo.PlayerName)
      PositionEntry.ColorBig = GoldColor;
    else
      PositionEntry.ColorBig = ColorStatsBig[PositionEntry.Column];

    CountRows[PositionEntry.Column]++;

    if (Entries[IndexEntry].Status == Status_Entering) {
      PositionEntryEntering = PositionEntry;
      PositionEntryEntering.FlagOffscreen = true;

      EntryMove(Entries[IndexEntry], PositionEntryEntering);
      Entries[IndexEntry].TimeInterpolation = Level.TimeSeconds;
      Entries[IndexEntry].Status = Status_Present;
      }

    EntryMove(Entries[IndexEntry], PositionEntry);
    }
  }


// ============================================================================
// Default Properties
// ============================================================================

defaultproperties
{
     TextBalanceNone="No team balance."
     TextBalanceLeader="Teams balanced by "
     TextBalanceTrailer="."
     TextBalanceType(1)="Size"
     TextBalanceType(2)="Clans"
     TextBalanceType(3)="Skill"
     TextBalanceBotSupported=" (Bot-Supported)"
     TextInfoSeparator="   "
     TextInfoOvertime="in Overtime"
     TextInfoCaptureCountdown="Capture Countdown:"
     TextInfoCaptureCountdownDisabled="(disabled)"
     TextInfoCaptureTimeout="Capture Timeout:"
     TextLocationArena="Arena"
     TextLocationJail(0)="Red Jail"
     TextLocationJail(1)="Blue Jail"
     TextLocationUndisclosed="Undisclosed"
     TextLocationUnknown="Unknown"
     TextStatsAttack="attack"
     TextStatsDefense="defense"
     TextStatsRelease="release"
     TextSortLeader="Players sorted by "
     TextSortTrailer1=".  Press ["
     TextSortTrailer2="] to change sort key.  "
     TextSortKey(0)="Score"
     TextSortKey(1)="Orders"
     TextSortKey(2)="Name"
     TextSortKey(3)="Ping"
     TextSortKey(4)="Attack Kills"
     TextSortKey(5)="Defense Kills"
     TextSortKey(6)="Released Teammates"
     FragGoal="Capture Limit:"
     WhiteColor=(R=224,G=224,B=224)
}
